<?php
defined('ABSPATH') || exit;
const DejavoWL_VERSION = '4.7.1';

if (class_exists('woocommerce') and dej_options('dej_show_wishlist' , '1')) {
    class DejavoWL
    {
        protected static $products = [];
        protected static $settings = [];
        protected static $localization = [];
        protected static $instance = null;

        public static function instance()
        {
            if (is_null(self::$instance)) {
                self::$instance = new self();
            }

            return self::$instance;
        }

        function __construct()
        {
            self::$settings = (array)get_option('DejavoWL_settings', []);
            self::$localization = (array)get_option('DejavoWL_localization', []);

            // add query var
            add_filter('query_vars', [$this, 'query_vars'], 1);
            add_action('init', [$this, 'init']);

            // menu
            add_action('admin_init', [$this, 'register_settings']);
            add_action('admin_menu', [$this, 'admin_menu']);

            // my account
            if (self::get_setting('page_myaccount', 'yes') === 'yes') {
                add_filter('woocommerce_account_menu_items', [$this, 'account_items'], 99);
                add_action('woocommerce_account_wishlist_endpoint', [$this, 'account_endpoint'], 99);
            }

            // scripts
            add_action('wp_enqueue_scripts', [$this, 'enqueue_scripts']);

            // quickview
            add_action('wp_ajax_wishlist_quickview', [$this, 'wishlist_quickview']);

            // add
            add_action('wp_ajax_wishlist_add', [$this, 'wishlist_add']);
            add_action('wp_ajax_nopriv_wishlist_add', [$this, 'wishlist_add']);

            // add to wishlist
            add_action('template_redirect', [$this, 'wishlist_add_by_link']);

            // added to cart
            if (self::get_setting('auto_remove', 'no') === 'yes') {
                add_action('woocommerce_add_to_cart', [$this, 'add_to_cart'], 10, 2);
            }

            // remove
            add_action('wp_ajax_wishlist_remove', [$this, 'wishlist_remove']);
            add_action('wp_ajax_nopriv_wishlist_remove', [$this, 'wishlist_remove']);

            // empty
            add_action('wp_ajax_wishlist_empty', [$this, 'wishlist_empty']);
            add_action('wp_ajax_nopriv_wishlist_empty', [$this, 'wishlist_empty']);

            // load
            add_action('wp_ajax_wishlist_load', [$this, 'wishlist_load']);
            add_action('wp_ajax_nopriv_wishlist_load', [$this, 'wishlist_load']);

            // load count
            add_action('wp_ajax_wishlist_load_count', [$this, 'wishlist_load_count']);
            add_action('wp_ajax_nopriv_wishlist_load_count', [$this, 'wishlist_load_count']);

            // fragments
            add_action('wp_ajax_DejavoWL_get_data', [$this, 'get_data']);
            add_action('wp_ajax_nopriv_DejavoWL_get_data', [$this, 'get_data']);


            // footer
            add_action('wp_footer', [$this, 'wp_footer']);

            // product columns
            add_filter('manage_edit-product_columns', [$this, 'product_columns'], 10);
            add_action('manage_product_posts_custom_column', [$this, 'posts_custom_column'], 10, 2);
            add_filter('manage_edit-product_sortable_columns', [$this, 'sortable_columns']);
            add_filter('request', [$this, 'request']);

            // post states
            add_filter('display_post_states', [$this, 'display_post_states'], 10, 2);

            // user login & logout
            add_action('wp_login', [$this, 'wp_login'], 10, 2);
            add_action('wp_logout', [$this, 'wp_logout']);

            // user columns
            add_filter('manage_users_columns', [$this, 'users_columns']);
            add_filter('manage_users_custom_column', [$this, 'users_columns_content'], 10, 3);

            // wpml
            add_filter('wcml_multi_currency_ajax_actions', [$this, 'wcml_multi_currency'], 99);

        }

        function query_vars($vars)
        {
            $vars[] = 'DejavoWL_id';

            return $vars;
        }

        function init()
        {
            // get key
            $key = isset($_COOKIE['DejavoWL_key']) ? sanitize_text_field($_COOKIE['DejavoWL_key']) : '#';

            // get products
            self::$products = self::get_ids($key);

            // rewrite
            if ($page_id = self::get_page_id()) {
                $page_slug = get_post_field('post_name', $page_id);

                if ($page_slug !== '') {
                    add_rewrite_rule('^' . $page_slug . '/([\w]+)/?', 'index.php?page_id=' . $page_id . '&DejavoWL_id=$matches[1]', 'top');
                    add_rewrite_rule('(.*?)/' . $page_slug . '/([\w]+)/?', 'index.php?page_id=' . $page_id . '&DejavoWL_id=$matches[2]', 'top');
                }
            }

            // my account page
            if (self::get_setting('page_myaccount', 'yes') === 'yes') {
                add_rewrite_endpoint('wishlist', EP_PAGES);
            }

            // shortcode
            add_shortcode('DejavoWL', [$this, 'wl_btn']);
            add_shortcode('DejavoWL_btn', [$this, 'wl_btn']);
            add_shortcode('DejavoWL_list', [$this, 'shortcode_list']);


        }

        public static function get_settings()
        {
            return apply_filters('DejavoWL_get_settings', self::$settings);
        }

        public static function get_setting($name, $default = false)
        {
            if (!empty(self::$settings) && isset(self::$settings[$name])) {
                $setting = self::$settings[$name];
            } else {
                $setting = get_option('DejavoWL_' . $name, $default);
            }

            return apply_filters('DejavoWL_get_setting', $setting, $name, $default);
        }

        public static function localization($key = '', $default = '')
        {
            $str = '';

            if (!empty($key) && !empty(self::$localization[$key])) {
                $str = self::$localization[$key];
            } elseif (!empty($default)) {
                $str = $default;
            }

            return esc_html(apply_filters('DejavoWL_localization_' . $key, $str));
        }

        function add_to_cart($cart_item_key, $product_id)
        {
            $key = self::get_key();

            if ($key !== '#') {
                $products = self::get_ids($key);

                if (array_key_exists($product_id, $products)) {
                    unset($products[$product_id]);
                    update_option('DejavoWL_list_' . $key, $products);
                    self::update_product_count($product_id, 'remove');
                }
            }
        }

        function wishlist_add_by_link()
        {
            if (!isset($_REQUEST['add-to-wishlist']) && !isset($_REQUEST['add_to_wishlist'])) {
                return false;
            }

            $key = self::get_key();
            $product_id = absint(isset($_REQUEST['add_to_wishlist']) ? (int)sanitize_text_field($_REQUEST['add_to_wishlist']) : 0);
            $product_id = absint(isset($_REQUEST['add-to-wishlist']) ? (int)sanitize_text_field($_REQUEST['add-to-wishlist']) : $product_id);

            if ($product_id) {
                if ($key !== '#' && $key !== 'DejavoWL') {
                    $product = wc_get_product($product_id);
                    $products = self::get_ids($key);

                    if (!array_key_exists($product_id, $products)) {
                        // insert if not exists
                        $products = [
                                $product_id => [
                                    'time' => time(),
                                    'price' => is_a($product, 'WC_Product') ? $product->get_price() : 0,
                                    'parent' => wp_get_post_parent_id($product_id) ?: 0,
                                ]
                            ] + $products;
                        update_option('DejavoWL_list_' . $key, $products);
                    }
                }
            }

            // redirect to wishlist page
            wp_safe_redirect(self::get_url($key, true));
        }

        function wishlist_add()
        {
            $return = ['status' => 0];
            $key = self::get_key();

            if (($product_id = (int)sanitize_text_field($_POST['product_id'])) > 0) {
                if ($key === '#') {
                    $return['status'] = 0;
                    $return['notice'] = self::localization('login_message', esc_html__('Please log in to use the Wishlist!', 'dejavo'));
                    $return['content'] = self::wishlist_content($key, self::localization('empty_message', esc_html__('There are no products on the Wishlist!', 'dejavo')));
                } else {
                    $products = self::get_ids($key);

                    if (!array_key_exists($product_id, $products)) {
                        // insert if not exists
                        $product = wc_get_product($product_id);
                        $products = [
                                $product_id => [
                                    'time' => time(),
                                    'price' => is_a($product, 'WC_Product') ? $product->get_price() : 0,
                                    'parent' => wp_get_post_parent_id($product_id) ?: 0,
                                ]
                            ] + $products;
                        update_option('DejavoWL_list_' . $key, $products);
                        self::update_product_count($product_id, 'add');
                        $return['notice'] = self::localization('added_message', esc_html__('{name} has been added to Wishlist.', 'dejavo'));
                    } else {
                        $return['notice'] = self::localization('already_message', esc_html__('{name} is already in the Wishlist.', 'dejavo'));
                    }

                    $return['status'] = 1;
                    $return['count'] = count($products);
                    $return['data'] = [
                        'key' => self::get_key(),
                        'ids' => self::get_ids(),
                    ];

                    if (self::get_setting('button_action', 'list') === 'list') {
                        $return['content'] = self::wishlist_content($key);
                    }
                }
            } else {
                $product_id = 0;
                $return['status'] = 0;
                $return['notice'] = self::localization('error_message', esc_html__('Have an error, please try again!', 'dejavo'));
            }

            do_action('DejavoWL_add', $product_id, $key);

            wp_send_json($return);
        }

        function wishlist_remove()
        {
            $return = ['status' => 0];
            $key = sanitize_text_field($_POST['key']);

            if (empty($key)) {
                $key = self::get_key();
            }

            if (($product_id = (int)sanitize_text_field($_POST['product_id'])) > 0) {
                if ($key === '#') {
                    $return['notice'] = self::localization('login_message', esc_html__('Please log in to use the Wishlist!', 'dejavo'));
                } else {
                    $products = self::get_ids($key);

                    if (array_key_exists($product_id, $products)) {
                        unset($products[$product_id]);
                        update_option('DejavoWL_list_' . $key, $products);
                        self::update_product_count($product_id, 'remove');
                        $return['count'] = count($products);
                        $return['status'] = 1;
                        $return['notice'] = self::localization('removed_message', esc_html__('Product has been removed from the Wishlist.', 'dejavo'));
                        $return['data'] = [
                            'key' => self::get_key(),
                            'ids' => self::get_ids(),
                        ];

                        if (empty($products)) {
                            $return['content'] = self::wishlist_content($key, self::localization('empty_message', esc_html__('There are no products on the Wishlist!', 'dejavo'))) . '</div>';
                        }
                    } else {
                        $return['notice'] = self::localization('not_exist_message', esc_html__('The product does not exist on the Wishlist!', 'dejavo'));
                    }
                }
            } else {
                $product_id = 0;
                $return['notice'] = self::localization('error_message', esc_html__('Have an error, please try again!', 'dejavo'));
            }

            do_action('DejavoWL_remove', $product_id, $key);

            wp_send_json($return);
        }

        function wishlist_empty()
        {
            $return = ['status' => 0];
            $key = sanitize_text_field($_POST['key']);

            if (empty($key)) {
                $key = self::get_key();
            }

            if ($key === '#') {
                $return['notice'] = self::localization('login_message', esc_html__('Please log in to use the Wishlist!', 'dejavo'));
            } else {
                if (($products = self::get_ids($key)) && !empty($products)) {
                    foreach (array_keys($products) as $product_id) {
                        // update count
                        self::update_product_count($product_id, 'remove');
                    }
                }

                // remove option
                update_option('DejavoWL_list_' . $key, []);
                $return['status'] = 1;
                $return['count'] = 0;
                $return['notice'] = self::localization('empty_notice', esc_html__('All products have been removed from the Wishlist!', 'dejavo'));
                $return['content'] = self::wishlist_content($key, self::localization('empty_message', esc_html__('There are no products on the Wishlist!', 'dejavo')));
                $return['data'] = [
                    'key' => self::get_key(),
                    'ids' => self::get_ids(),
                ];
            }

            do_action('DejavoWL_empty', $key);

            wp_send_json($return);
        }

        function wishlist_load()
        {
            $return = ['status' => 0];
            $key = self::get_key();

            if ($key === '#') {
                $return['notice'] = self::localization('login_message', esc_html__('Please log in to use Wishlist!', 'dejavo'));
                $return['content'] = self::wishlist_content($key, self::localization('empty_message', esc_html__('There are no products on the Wishlist!', 'dejavo')));
            } else {
                $products = self::get_ids($key);
                $return['status'] = 1;
                $return['count'] = count($products);
                $return['content'] = self::wishlist_content($key);
                $return['data'] = [
                    'key' => self::get_key(),
                    'ids' => self::get_ids(),
                ];
            }

            do_action('DejavoWL_load', $key);

            wp_send_json($return);
        }

        function wishlist_load_count()
        {
            $return = ['status' => 0, 'count' => 0];
            $key = self::get_key();

            if ($key === '#') {
                $return['notice'] = self::localization('login_message', esc_html__('Please log in to use Wishlist!', 'dejavo'));
            } else {
                $products = self::get_ids($key);
                $return['status'] = 1;
                $return['count'] = count($products);
            }

            do_action('wishlist_load_count', $key);

            wp_send_json($return);
        }


        function manage_wishlists()
        {
            ob_start();
            self::manage_content();
            echo ob_get_clean();
            wp_die();
        }

        function add_wishlist()
        {
            $name = trim(isset($_POST['name']) ? sanitize_text_field($_POST['name']) : '');

            if ($user_id = get_current_user_id()) {
                $key = self::get_key(true);
                $keys = get_user_meta($user_id, 'DejavoWL_keys', true) ?: [];
                $max = self::get_setting('maximum_wishlists', '5');

                if (is_array($keys) && (count($keys) < (int)$max)) {
                    $keys[$key] = [
                        'name' => $name,
                        'time' => time()
                    ];

                    update_user_meta($user_id, 'DejavoWL_keys', $keys);
                }

                ob_start();
                self::manage_content();
                echo ob_get_clean();
            }

            wp_die();
        }

        function delete_wishlist()
        {
            $key = trim(isset($_POST['key']) ? sanitize_text_field($_POST['key']) : '');

            if (!empty($key) && ($user_id = get_current_user_id())) {
                // delete key from user
                $keys = get_user_meta($user_id, 'DejavoWL_keys', true) ?: [];

                if (is_array($keys) && (count($keys) > 1)) {
                    // don't remove primary key
                    unset($keys[$key]);
                    update_user_meta($user_id, 'DejavoWL_keys', $keys);

                    // delete wishlist
                    delete_option('DejavoWL_list_' . $key);
                }

                ob_start();
                self::manage_content();
                echo ob_get_clean();
            }

            wp_die();
        }

        function view_wishlist()
        {
            $key = trim(isset($_POST['key']) ? sanitize_text_field($_POST['key']) : '');

            if (!empty($key)) {
                echo self::wishlist_content($key);
            }

            wp_die();
        }

        function set_default()
        {
            $return = [];
            $key = trim(isset($_POST['key']) ? sanitize_text_field($_POST['key']) : '');
            $products = self::get_ids($key);
            $count = count($products);

            if (!empty($key) && ($user_id = get_current_user_id())) {
                update_user_meta($user_id, 'DejavoWL_key', $key);

                // set cookie
                $secure = apply_filters('DejavoWL_cookie_secure', wc_site_is_https() && is_ssl());
                $httponly = apply_filters('DejavoWL_cookie_httponly', false);

                wc_setcookie('DejavoWL_key', $key, time() + 604800, $secure, $httponly);

                ob_start();
                self::manage_content();
                $return['content'] = ob_get_clean();
                $return['count'] = $count;
                $return['products'] = array_keys($products);
                $return['data'] = [
                    'key' => self::get_key(),
                    'ids' => self::get_ids(),
                ];
            }

            wp_send_json($return);
        }

        function add_button()
        {
            echo do_shortcode('[DejavoWL]');
        }

        public static function wl_btn($attrs): string
        {
            $output = $product_name = $product_image = '';

            $attrs = shortcode_atts([
                'id' => null,
                'type' => 'button'
            ], $attrs, 'DejavoWL');

            if (!$attrs['id']) {
                global $product;

                if ($product && is_a($product, 'WC_Product')) {
                    $attrs['id'] = $product->get_id();
                    $product_name = $product->get_name();
                    $product_image_id = $product->get_image_id();
                    $product_image = wp_get_attachment_image_url($product_image_id);
                }
            } else {
                if ($_product = wc_get_product($attrs['id'])) {
                    $product_name = $_product->get_name();
                    $product_image_id = $_product->get_image_id();
                    $product_image = wp_get_attachment_image_url($product_image_id);
                }
            }

            if ($attrs['id']) {

                $class = 'DejavoWL-btn DejavoWL-btn-' . esc_attr($attrs['id']);

                if (array_key_exists($attrs['id'], self::$products) || in_array($attrs['id'], array_column(self::$products, 'parent'))) {
                    $class .= ' DejavoWL-added';
                }


                $output = '<a href="' . esc_url('?add-to-wishlist=' . $attrs['id']) . '" class="' . esc_attr($class) . '" data-id="' . esc_attr($attrs['id']) . '" data-product_name="' . esc_attr($product_name) . '" data-product_image="' . esc_attr($product_image) . '"><i class="dej-heart"></i></a>';

            }

            return wp_kses_post(apply_filters('DejavoWL_button_html', $output, $attrs['id'], $attrs));
        }

        function shortcode_list($attrs)
        {
            $attrs = shortcode_atts([
                'key' => null
            ], $attrs, 'DejavoWL_list');

            if (!empty($attrs['key'])) {
                $key = $attrs['key'];
            } else {
                if (get_query_var('DejavoWL_id')) {
                    $key = get_query_var('DejavoWL_id');
                } elseif (isset($_REQUEST['wid']) && !empty($_REQUEST['wid'])) {
                    $key = sanitize_text_field($_REQUEST['wid']);
                } else {
                    $key = self::get_key();
                }
            }

            if (get_query_var('DejavoWL_id')) {
                $key = get_query_var('DejavoWL_id');
            } elseif (isset($_REQUEST['wid']) && !empty($_REQUEST['wid'])) {
                $key = sanitize_text_field($_REQUEST['wid']);
            } else {
                $key = self::get_key();
            }


            $share_url_raw = self::get_url($key, true);
            $share_url = urlencode($share_url_raw);
            $return_html = '<div class="DejavoWL-list">';
            $return_html .= self::get_items($key, 'table');
            $return_html .= '<div class="DejavoWL-actions">';


            $return_html .= '<div class="DejavoWL-share">';
            $return_html .= '<span class="DejavoWL-share-label">' . esc_html__('Share on:', 'dejavo') . '</span>';
            $return_html .= '<a class="DejavoWL-share-facebook" href="https://www.facebook.com/sharer.php?u=' . $share_url . '" target="_blank"><i class="dejbrand-facebook"></i></a>';
            $return_html .= '<a class="DejavoWL-share-twitter" href="https://twitter.com/share?url=' . $share_url . '" target="_blank"><i class="dejbrand-twitter"></i></a>';
            $return_html .= '<a class="DejavoWL-share-pinterest" href="https://pinterest.com/pin/create/button/?url=' . $share_url . '" target="_blank"><i class="dejbrand-pinterest"></i></a>';
            $return_html .= '<a class="DejavoWL-share-mail" href="mailto:?body=' . $share_url . '" target="_blank"><i class="dej-envelope"></i></a>';
            $return_html .= '</div><!-- /DejavoWL-share -->';


            $return_html .= '<div class="DejavoWL-copy">';
            $return_html .= '<span class="DejavoWL-copy-label">' . esc_html__('Wishlist link:', 'dejavo') . '</span>';
            $return_html .= '<span class="DejavoWL-copy-url"><input id="DejavoWL_copy_url" type="url" value="' . esc_attr($share_url_raw) . '" readonly/></span>';
            $return_html .= '<span class="DejavoWL-copy-btn"><input id="DejavoWL_copy_btn" type="button" value="' . esc_html__('Copy', 'dejavo') . '"/></span>';
            $return_html .= '</div><!-- /DejavoWL-copy -->';


            $return_html .= '</div><!-- /DejavoWL-actions -->';
            $return_html .= '</div><!-- /DejavoWL-list -->';

            return $return_html;
        }

        function register_settings()
        {
            // settings
            register_setting('DejavoWL_settings', 'DejavoWL_settings');

            // localization
            register_setting('DejavoWL_localization', 'DejavoWL_localization');
        }

        function admin_menu()
        {
            add_submenu_page('future', __('Wishlist', 'dejavo'), __('Wishlist', 'dejavo'), 'manage_options', 'future-DejavoWL', array(
                $this,
                'admin_menu_content'
            ));
        }

        function admin_menu_content()
        {
            $active_tab = isset($_GET['tab']) ? sanitize_key($_GET['tab']) : 'settings';
            ?>
            <div class="dejavo_settings_page wrap">
                <h1 class="dejavo_settings_page_title"><?php echo esc_html_e('Wishlist', 'dejavo') . ' ' . DejavoWL_VERSION; ?></h1>

                <?php if (isset($_GET['settings-updated']) && $_GET['settings-updated']) { ?>
                    <div class="notice notice-success is-dismissible">
                        <p><?php esc_html_e('Settings updated.', 'dejavo'); ?></p>
                    </div>
                <?php } ?>
                <div class="dejavo_settings_page_nav">
                    <h2 class="nav-tab-wrapper">
                        <a href="<?php echo admin_url('admin.php?page=future-DejavoWL&tab=settings'); ?>"
                           class="<?php echo esc_attr($active_tab === 'settings' ? 'nav-tab nav-tab-active' : 'nav-tab'); ?>">
                            <?php esc_html_e('Settings', 'dejavo'); ?>
                        </a>
                        <a href="<?php echo admin_url('admin.php?page=future-DejavoWL&tab=localization'); ?>"
                           class="<?php echo esc_attr($active_tab === 'localization' ? 'nav-tab nav-tab-active' : 'nav-tab'); ?>">
                            <?php esc_html_e('Localization', 'dejavo'); ?>
                        </a>
                    </h2>
                </div>
                <div class="dejavo_settings_page_content">
                    <?php if ($active_tab === 'settings') {
                        if (isset($_REQUEST['settings-updated']) && (sanitize_text_field($_REQUEST['settings-updated']) === 'true')) {
                            flush_rewrite_rules();
                        }

                        $disable_unauthenticated = self::get_setting('disable_unauthenticated', 'no');
                        $auto_remove = self::get_setting('auto_remove', 'no');
                        $button_action = self::get_setting('button_action', 'list');
                        $message_position = self::get_setting('message_position', 'right-top');
                        $button_action_added = self::get_setting('button_action_added', 'popup');
                        $popup_position = self::get_setting('popup_position', 'center');
                        $link = self::get_setting('link', 'yes');
                        $show_price_change = self::get_setting('show_price_change', 'both');
                        $empty_button = self::get_setting('empty_button', 'yes');
                        $page_myaccount = self::get_setting('page_myaccount', 'yes');
                        $menu_action = self::get_setting('menu_action', 'open_popup');
                        ?>
                        <form method="post" action="options.php">
                            <table class="form-table">
                                <tr class="heading">
                                    <th colspan="2">
                                        <?php esc_html_e('General', 'dejavo'); ?>
                                    </th>
                                </tr>
                                <tr>
                                    <th scope="row"><?php esc_html_e('Disable the wishlist for unauthenticated users', 'dejavo'); ?></th>
                                    <td>
                                        <select name="DejavoWL_settings[disable_unauthenticated]">
                                            <option value="yes" <?php selected($disable_unauthenticated, 'yes'); ?>><?php esc_html_e('Yes', 'dejavo'); ?></option>
                                            <option value="no" <?php selected($disable_unauthenticated, 'no'); ?>><?php esc_html_e('No', 'dejavo'); ?></option>
                                        </select>
                                    </td>
                                </tr>
                                <tr>
                                    <th scope="row"><?php esc_html_e('Auto remove', 'dejavo'); ?></th>
                                    <td>
                                        <select name="DejavoWL_settings[auto_remove]">
                                            <option value="yes" <?php selected($auto_remove, 'yes'); ?>><?php esc_html_e('Yes', 'dejavo'); ?></option>
                                            <option value="no" <?php selected($auto_remove, 'no'); ?>><?php esc_html_e('No', 'dejavo'); ?></option>
                                        </select>
                                        <span class="description"><?php esc_html_e('Auto remove product from the wishlist after adding to the cart.', 'dejavo'); ?></span>
                                    </td>
                                </tr>


                                <tr class="heading">
                                    <th>
                                        <?php esc_html_e('Button', 'dejavo'); ?>
                                    </th>
                                    <td>
                                        <?php esc_html_e('Settings for "Add to wishlist" button.', 'dejavo'); ?>
                                    </td>
                                </tr>


                                <tr>
                                    <th scope="row"><?php esc_html_e('Action', 'dejavo'); ?></th>
                                    <td>
                                        <select name="DejavoWL_settings[button_action]" class="DejavoWL_button_action">
                                            <option value="message" <?php selected($button_action, 'message'); ?>><?php esc_html_e('Show message', 'dejavo'); ?></option>
                                            <option value="list" <?php selected($button_action, 'list'); ?>><?php esc_html_e('Open wishlist popup', 'dejavo'); ?></option>
                                            <option value="no" <?php selected($button_action, 'no'); ?>><?php esc_html_e('Add to wishlist solely', 'dejavo'); ?></option>
                                        </select>
                                        <span class="description"><?php esc_html_e('Action triggered by clicking on the wishlist button.', 'dejavo'); ?></span>
                                    </td>
                                </tr>
                                <tr class="DejavoWL_button_action_hide DejavoWL_button_action_message">
                                    <th scope="row"><?php esc_html_e('Message position', 'dejavo'); ?></th>
                                    <td>
                                        <select name="DejavoWL_settings[message_position]">
                                            <option value="right-top" <?php selected($message_position, 'right-top'); ?>><?php esc_html_e('right-top', 'dejavo'); ?></option>
                                            <option value="right-bottom" <?php selected($message_position, 'right-bottom'); ?>><?php esc_html_e('right-bottom', 'dejavo'); ?></option>
                                            <option value="fluid-top" <?php selected($message_position, 'fluid-top'); ?>><?php esc_html_e('center-top', 'dejavo'); ?></option>
                                            <option value="fluid-bottom" <?php selected($message_position, 'fluid-bottom'); ?>><?php esc_html_e('center-bottom', 'dejavo'); ?></option>
                                            <option value="left-top" <?php selected($message_position, 'left-top'); ?>><?php esc_html_e('left-top', 'dejavo'); ?></option>
                                            <option value="left-bottom" <?php selected($message_position, 'left-bottom'); ?>><?php esc_html_e('left-bottom', 'dejavo'); ?></option>
                                        </select>
                                    </td>
                                </tr>
                                <tr>
                                    <th scope="row"><?php esc_html_e('Action (added)', 'dejavo'); ?></th>
                                    <td>
                                        <select name="DejavoWL_settings[button_action_added]">
                                            <option value="popup" <?php selected($button_action_added, 'popup'); ?>><?php esc_html_e('Open wishlist popup', 'dejavo'); ?></option>
                                            <option value="page" <?php selected($button_action_added, 'page'); ?>><?php esc_html_e('Open wishlist page', 'dejavo'); ?></option>
                                        </select>
                                        <span class="description"><?php esc_html_e('Action triggered by clicking on the wishlist button after adding an item to the wishlist.', 'dejavo'); ?></span>
                                    </td>
                                </tr>


                                <tr>
                                    <th scope="row"><?php esc_html_e('Shortcode', 'dejavo'); ?></th>
                                    <td>
                                                <span class="description">
                                                    <?php printf(esc_html__('You can add a button manually by using the shortcode %s, eg. %s for the product whose ID is 99.', 'dejavo'), '<code>[DejavoWL id="{product id}"]</code>', '<code>[DejavoWL id="99"]</code>'); ?>
                                                </span>
                                    </td>
                                </tr>

                                <tr class="heading">
                                    <th>
                                        <?php esc_html_e('Popup', 'dejavo'); ?>
                                    </th>
                                    <td>
                                        <?php esc_html_e('Settings for the wishlist popup.', 'dejavo'); ?>
                                    </td>
                                </tr>
                                <tr>
                                    <th scope="row"><?php esc_html_e('Position', 'dejavo'); ?></th>
                                    <td>
                                        <select name="DejavoWL_settings[popup_position]">
                                            <option value="center" <?php selected($popup_position, 'center'); ?>><?php esc_html_e('Center', 'dejavo'); ?></option>
                                            <option value="right" <?php selected($popup_position, 'right'); ?>><?php esc_html_e('Right', 'dejavo'); ?></option>
                                            <option value="left" <?php selected($popup_position, 'left'); ?>><?php esc_html_e('Left', 'dejavo'); ?></option>
                                        </select>
                                    </td>
                                </tr>

                                <tr>
                                    <th><?php esc_html_e('Link to individual product', 'dejavo'); ?></th>
                                    <td>
                                        <select name="DejavoWL_settings[link]">
                                            <option value="yes" <?php selected($link, 'yes'); ?>><?php esc_html_e('Yes, open in the same tab', 'dejavo'); ?></option>
                                            <option value="yes_blank" <?php selected($link, 'yes_blank'); ?>><?php esc_html_e('Yes, open in the new tab', 'dejavo'); ?></option>
                                            <option value="yes_popup" <?php selected($link, 'yes_popup'); ?>><?php esc_html_e('Yes, open quick view popup', 'dejavo'); ?></option>
                                            <option value="no" <?php selected($link, 'no'); ?>><?php esc_html_e('No', 'dejavo'); ?></option>
                                        </select>
                                    </td>
                                </tr>
                                <tr>
                                    <th scope="row"><?php esc_html_e('Show price change', 'dejavo'); ?></th>
                                    <td>
                                        <select name="DejavoWL_settings[show_price_change]">
                                            <option value="no" <?php selected($show_price_change, 'no'); ?>><?php esc_html_e('No', 'dejavo'); ?></option>
                                            <option value="increase" <?php selected($show_price_change, 'increase'); ?>><?php esc_html_e('Increase only', 'dejavo'); ?></option>
                                            <option value="decrease" <?php selected($show_price_change, 'decrease'); ?>><?php esc_html_e('Decrease only', 'dejavo'); ?></option>
                                            <option value="both" <?php selected($show_price_change, 'both'); ?>><?php esc_html_e('Both increase and decrease', 'dejavo'); ?></option>
                                        </select>
                                        <span class="description"><?php esc_html_e('Show price change since a product was added.', 'dejavo'); ?></span>
                                    </td>
                                </tr>


                                <tr>
                                    <th scope="row"><?php esc_html_e('Empty wishlist button', 'dejavo'); ?></th>
                                    <td>
                                        <select name="DejavoWL_settings[empty_button]">
                                            <option value="yes" <?php selected($empty_button, 'yes'); ?>><?php esc_html_e('Yes', 'dejavo'); ?></option>
                                            <option value="no" <?php selected($empty_button, 'no'); ?>><?php esc_html_e('No', 'dejavo'); ?></option>
                                        </select>
                                        <span class="description"><?php esc_html_e('Show empty wishlist button on the popup?', 'dejavo'); ?></span>
                                    </td>
                                </tr>
                                <tr>
                                    <th scope="row"><?php esc_html_e('Continue shopping link', 'dejavo'); ?></th>
                                    <td>
                                        <input type="url" name="DejavoWL_settings[continue_url]"
                                               value="<?php echo esc_attr(self::get_setting('continue_url')); ?>"
                                               class="regular-text code"/>
                                        <span class="description"><?php esc_html_e('By default, the wishlist popup will only be closed when customers click on the "Continue Shopping" button.', 'dejavo'); ?></span>
                                    </td>
                                </tr>
                                <tr class="heading">
                                    <th>
                                        <?php esc_html_e('Page', 'dejavo'); ?>
                                    </th>
                                    <td>
                                        <?php esc_html_e('Settings for wishlist page.', 'dejavo'); ?>
                                    </td>
                                </tr>
                                <tr>
                                    <th scope="row"><?php esc_html_e('Wishlist page', 'dejavo'); ?></th>
                                    <td>
                                        <?php wp_dropdown_pages([
                                            'selected' => self::get_setting('page_id', ''),
                                            'name' => 'DejavoWL_settings[page_id]',
                                            'show_option_none' => esc_html__('Choose a page', 'dejavo'),
                                            'option_none_value' => '',
                                        ]); ?>
                                        <span class="description"><?php printf(esc_html__('Add shortcode %s to display the wishlist on a page.', 'dejavo'), '<code>[DejavoWL_list]</code>'); ?></span>
                                    </td>
                                </tr>

                                <tr>
                                    <th scope="row"><?php esc_html_e('Add Wishlist page to My Account', 'dejavo'); ?></th>
                                    <td>
                                        <select name="DejavoWL_settings[page_myaccount]">
                                            <option value="yes" <?php selected($page_myaccount, 'yes'); ?>><?php esc_html_e('Yes', 'dejavo'); ?></option>
                                            <option value="no" <?php selected($page_myaccount, 'no'); ?>><?php esc_html_e('No', 'dejavo'); ?></option>
                                        </select>
                                    </td>
                                </tr>

                                <tr>
                                    <th scope="row"><?php esc_html_e('Action', 'dejavo'); ?></th>
                                    <td>
                                        <select name="DejavoWL_settings[menu_action]">
                                            <option value="open_page" <?php selected($menu_action, 'open_page'); ?>><?php esc_html_e('Open wishlist page', 'dejavo'); ?></option>
                                            <option value="open_popup" <?php selected($menu_action, 'open_popup'); ?>><?php esc_html_e('Open wishlist popup', 'dejavo'); ?></option>
                                        </select>
                                        <span class="description"><?php esc_html_e('Action when clicking on the "wishlist menu".', 'dejavo'); ?></span>
                                    </td>
                                </tr>

                                <tr class="submit">
                                    <th colspan="2">
                                        <?php settings_fields('DejavoWL_settings'); ?><?php submit_button(); ?>
                                    </th>
                                </tr>
                            </table>
                        </form>
                    <?php } elseif ($active_tab === 'localization') { ?>
                        <form method="post" action="options.php">
                            <table class="form-table">
                                <tr class="heading">
                                    <th scope="row"><?php esc_html_e('Localization', 'dejavo'); ?></th>
                                    <td>
                                        <?php esc_html_e('Leave blank to use the default text and its equivalent translation in multiple languages.', 'dejavo'); ?>
                                    </td>
                                </tr>
                                <tr>
                                    <th><?php esc_html_e('Button text', 'dejavo'); ?></th>
                                    <td>
                                        <input type="text" class="regular-text" name="DejavoWL_localization[button]"
                                               value="<?php echo esc_attr(self::localization('button')); ?>"
                                               placeholder="<?php esc_attr_e('Add to wishlist', 'dejavo'); ?>"/>
                                    </td>
                                </tr>
                                <tr>
                                    <th><?php esc_html_e('Button text (added)', 'dejavo'); ?></th>
                                    <td>
                                        <input type="text" class="regular-text"
                                               name="DejavoWL_localization[button_added]"
                                               value="<?php echo esc_attr(self::localization('button_added')); ?>"
                                               placeholder="<?php esc_attr_e('Browse wishlist', 'dejavo'); ?>"/>
                                    </td>
                                </tr>
                                <tr>
                                    <th><?php esc_html_e('Wishlist popup heading', 'dejavo'); ?></th>
                                    <td>
                                        <input type="text" class="regular-text"
                                               name="DejavoWL_localization[popup_heading]"
                                               value="<?php echo esc_attr(self::localization('popup_heading')); ?>"
                                               placeholder="<?php esc_attr_e('Wishlist', 'dejavo'); ?>"/>
                                    </td>
                                </tr>
                                <tr>
                                    <th><?php esc_html_e('Empty wishlist button', 'dejavo'); ?></th>
                                    <td>
                                        <input type="text" class="regular-text"
                                               name="DejavoWL_localization[empty_button]"
                                               value="<?php echo esc_attr(self::localization('empty_button')); ?>"
                                               placeholder="<?php esc_attr_e('remove all', 'dejavo'); ?>"/>
                                    </td>
                                </tr>

                                <tr>
                                    <th><?php esc_html_e('Price increase', 'dejavo'); ?></th>
                                    <td>
                                        <input type="text" class="regular-text"
                                               name="DejavoWL_localization[price_increase]"
                                               value="<?php echo esc_attr(self::localization('price_increase')); ?>"
                                               placeholder="<?php esc_attr_e('Increase {percentage} since added', 'dejavo'); ?>"/>
                                    </td>
                                </tr>
                                <tr>
                                    <th><?php esc_html_e('Price decrease', 'dejavo'); ?></th>
                                    <td>
                                        <input type="text" class="regular-text"
                                               name="DejavoWL_localization[price_decrease]"
                                               value="<?php echo esc_attr(self::localization('price_decrease')); ?>"
                                               placeholder="<?php esc_attr_e('Decrease {percentage} since added', 'dejavo'); ?>"/>
                                    </td>
                                </tr>
                                <tr>
                                    <th><?php esc_html_e('Open wishlist page', 'dejavo'); ?></th>
                                    <td>
                                        <input type="text" class="regular-text" name="DejavoWL_localization[open_page]"
                                               value="<?php echo esc_attr(self::localization('open_page')); ?>"
                                               placeholder="<?php esc_attr_e('Open wishlist page', 'dejavo'); ?>"/>
                                    </td>
                                </tr>
                                <tr>
                                    <th><?php esc_html_e('Continue shopping', 'dejavo'); ?></th>
                                    <td>
                                        <input type="text" class="regular-text" name="DejavoWL_localization[continue]"
                                               value="<?php echo esc_attr(self::localization('continue')); ?>"
                                               placeholder="<?php esc_attr_e('Continue shopping', 'dejavo'); ?>"/>
                                    </td>
                                </tr>


                                <tr class="heading">
                                    <th scope="row"><?php esc_html_e('Multiple Wishlist', 'dejavo'); ?></th>
                                    <td></td>
                                </tr>
                                <tr>
                                    <th scope="row"><?php esc_html_e('Primary wishlist name', 'dejavo'); ?></th>
                                    <td>
                                        <input type="text" name="DejavoWL_localization[primary_name]"
                                               class="regular-text"
                                               value="<?php echo esc_attr(self::localization('primary_name')); ?>"
                                               placeholder="<?php esc_attr_e('Wishlist', 'dejavo'); ?>"/>
                                    </td>
                                </tr>
                                <tr>
                                    <th scope="row"><?php esc_html_e('Manage wishlists', 'dejavo'); ?></th>
                                    <td>
                                        <input type="text" name="DejavoWL_localization[manage_wishlists]"
                                               class="regular-text"
                                               value="<?php echo esc_attr(self::localization('manage_wishlists')); ?>"
                                               placeholder="<?php esc_attr_e('Manage wishlists', 'dejavo'); ?>"/>
                                    </td>
                                </tr>
                                <tr>
                                    <th scope="row"><?php esc_html_e('Set default', 'dejavo'); ?></th>
                                    <td>
                                        <input type="text" name="DejavoWL_localization[set_default]"
                                               class="regular-text"
                                               value="<?php echo esc_attr(self::localization('set_default')); ?>"
                                               placeholder="<?php esc_attr_e('set default', 'dejavo'); ?>"/>
                                    </td>
                                </tr>
                                <tr>
                                    <th scope="row"><?php esc_html_e('Default', 'dejavo'); ?></th>
                                    <td>
                                        <input type="text" name="DejavoWL_localization[is_default]" class="regular-text"
                                               value="<?php echo esc_attr(self::localization('is_default')); ?>"
                                               placeholder="<?php esc_attr_e('default', 'dejavo'); ?>"/>
                                    </td>
                                </tr>
                                <tr>
                                    <th scope="row"><?php esc_html_e('Delete', 'dejavo'); ?></th>
                                    <td>
                                        <input type="text" name="DejavoWL_localization[delete]" class="regular-text"
                                               value="<?php echo esc_attr(self::localization('delete')); ?>"
                                               placeholder="<?php esc_attr_e('delete', 'dejavo'); ?>"/>
                                    </td>
                                </tr>
                                <tr>
                                    <th scope="row"><?php esc_html_e('Wishlist name placeholder', 'dejavo'); ?></th>
                                    <td>
                                        <input type="text" name="DejavoWL_localization[placeholder_name]"
                                               class="regular-text"
                                               value="<?php echo esc_attr(self::localization('placeholder_name')); ?>"
                                               placeholder="<?php esc_attr_e('New Wishlist', 'dejavo'); ?>"/>
                                    </td>
                                </tr>
                                <tr>
                                    <th scope="row"><?php esc_html_e('Add new wishlist', 'dejavo'); ?></th>
                                    <td>
                                        <input type="text" name="DejavoWL_localization[add_wishlist]"
                                               class="regular-text"
                                               value="<?php echo esc_attr(self::localization('add_wishlist')); ?>"
                                               placeholder="<?php esc_attr_e('Add New Wishlist', 'dejavo'); ?>"/>
                                    </td>
                                </tr>
                                <tr class="heading">
                                    <th scope="row"><?php esc_html_e('Message', 'dejavo'); ?></th>
                                    <td></td>
                                </tr>
                                <tr>
                                    <th><?php esc_html_e('Added to the wishlist', 'dejavo'); ?></th>
                                    <td>
                                        <input type="text" class="regular-text"
                                               name="DejavoWL_localization[added_message]"
                                               value="<?php echo esc_attr(self::localization('added_message')); ?>"
                                               placeholder="<?php esc_attr_e('{name} has been added to Wishlist.', 'dejavo'); ?>"/>
                                    </td>
                                </tr>
                                <tr>
                                    <th><?php esc_html_e('Already in the wishlist', 'dejavo'); ?></th>
                                    <td>
                                        <input type="text" class="regular-text"
                                               name="DejavoWL_localization[already_message]"
                                               value="<?php echo esc_attr(self::localization('already_message')); ?>"
                                               placeholder="<?php esc_attr_e('{name} is already in the Wishlist.', 'dejavo'); ?>"/>
                                    </td>
                                </tr>
                                <tr>
                                    <th><?php esc_html_e('Removed from wishlist', 'dejavo'); ?></th>
                                    <td>
                                        <input type="text" class="regular-text"
                                               name="DejavoWL_localization[removed_message]"
                                               value="<?php echo esc_attr(self::localization('removed_message')); ?>"
                                               placeholder="<?php esc_attr_e('Product has been removed from the Wishlist.', 'dejavo'); ?>"/>
                                    </td>
                                </tr>
                                <tr>
                                    <th><?php esc_html_e('Empty wishlist confirm', 'dejavo'); ?></th>
                                    <td>
                                        <input type="text" class="regular-text"
                                               name="DejavoWL_localization[empty_confirm]"
                                               value="<?php echo esc_attr(self::localization('empty_confirm')); ?>"
                                               placeholder="<?php esc_attr_e('This action cannot be undone. Are you sure?', 'dejavo'); ?>"/>
                                    </td>
                                </tr>
                                <tr>
                                    <th><?php esc_html_e('Empty wishlist notice', 'dejavo'); ?></th>
                                    <td>
                                        <input type="text" class="regular-text"
                                               name="DejavoWL_localization[empty_notice]"
                                               value="<?php echo esc_attr(self::localization('empty_notice')); ?>"
                                               placeholder="<?php esc_attr_e('All products have been removed from the Wishlist!', 'dejavo'); ?>"/>
                                    </td>
                                </tr>
                                <tr>
                                    <th><?php esc_html_e('Empty wishlist', 'dejavo'); ?></th>
                                    <td>
                                        <input type="text" class="regular-text"
                                               name="DejavoWL_localization[empty_message]"
                                               value="<?php echo esc_attr(self::localization('empty_message')); ?>"
                                               placeholder="<?php esc_attr_e('There are no products on the Wishlist!', 'dejavo'); ?>"/>
                                    </td>
                                </tr>
                                <tr>
                                    <th><?php esc_html_e('Delete wishlist confirm', 'dejavo'); ?></th>
                                    <td>
                                        <input type="text" class="regular-text"
                                               name="DejavoWL_localization[delete_confirm]"
                                               value="<?php echo esc_attr(self::localization('delete_confirm')); ?>"
                                               placeholder="<?php esc_attr_e('This action cannot be undone. Are you sure?', 'dejavo'); ?>"/>
                                    </td>
                                </tr>
                                <tr>
                                    <th><?php esc_html_e('Product does not exist', 'dejavo'); ?></th>
                                    <td>
                                        <input type="text" class="regular-text"
                                               name="DejavoWL_localization[not_exist_message]"
                                               value="<?php echo esc_attr(self::localization('not_exist_message')); ?>"
                                               placeholder="<?php esc_attr_e('The product does not exist on the Wishlist!', 'dejavo'); ?>"/>
                                    </td>
                                </tr>
                                <tr>
                                    <th><?php esc_html_e('Need to login', 'dejavo'); ?></th>
                                    <td>
                                        <input type="text" class="regular-text"
                                               name="DejavoWL_localization[login_message]"
                                               value="<?php echo esc_attr(self::localization('login_message')); ?>"
                                               placeholder="<?php esc_attr_e('Please log in to use the Wishlist!', 'dejavo'); ?>"/>
                                    </td>
                                </tr>
                                <tr>
                                    <th><?php esc_html_e('Copied wishlist link', 'dejavo'); ?></th>
                                    <td>
                                        <input type="text" class="regular-text" name="DejavoWL_localization[copied]"
                                               value="<?php echo esc_attr(self::localization('copied')); ?>"
                                               placeholder="<?php esc_html_e('Copied the wishlist link:', 'dejavo'); ?>"/>
                                    </td>
                                </tr>
                                <tr>
                                    <th><?php esc_html_e('Have an error', 'dejavo'); ?></th>
                                    <td>
                                        <input type="text" class="regular-text"
                                               name="DejavoWL_localization[error_message]"
                                               value="<?php echo esc_attr(self::localization('error_message')); ?>"
                                               placeholder="<?php esc_attr_e('Have an error, please try again!', 'dejavo'); ?>"/>
                                    </td>
                                </tr>
                                <tr class="submit">
                                    <th colspan="2">
                                        <?php settings_fields('DejavoWL_localization'); ?><?php submit_button(); ?>
                                    </th>
                                </tr>
                            </table>
                        </form>
                    <?php } ?>
                </div>
            </div>
            <?php
        }

        function account_items($items)
        {
            if (isset($items['customer-logout'])) {
                $logout = $items['customer-logout'];
                unset($items['customer-logout']);
            } else {
                $logout = '';
            }

            if (!isset($items['wishlist'])) {
                $items['wishlist'] = apply_filters('DejavoWL_myaccount_wishlist_label', esc_html__('Wishlist', 'dejavo'));
            }

            if ($logout) {
                $items['customer-logout'] = $logout;
            }

            return $items;
        }

        function account_endpoint()
        {
            echo apply_filters('DejavoWL_myaccount_wishlist_content', do_shortcode('[DejavoWL_list]'));
        }

        function enqueue_scripts()
        {

            if (self::get_setting('button_action', 'list') === 'message') {
                wp_enqueue_style('notiny', DEJ_TUI . '/assets/libs/notiny/notiny.css');
                wp_enqueue_script('notiny', DEJ_TUI . '/assets/libs/notiny/notiny.js', ['jquery'], DejavoWL_VERSION, true);
            }

            // main style
            wp_enqueue_style('DejavoWL-frontend', DEJ_TUI . '/assets/css/wishlist.css', [], DejavoWL_VERSION);

            // main js
            wp_enqueue_script('DejavoWL-frontend', DEJ_TUI . '/assets/js/wishlist.min.js', [
                'jquery',
                'js-cookie'
            ], DejavoWL_VERSION, true);

            // localize
            wp_localize_script('DejavoWL-frontend', 'DejavoWL_vars', [
                    'ajax_url' => admin_url('admin-ajax.php'),
                    'menu_action' => self::get_setting('menu_action', 'open_popup'),
                    'wishlist_url' => self::get_url(),
                    'button_action' => self::get_setting('button_action', 'list'),
                    'message_position' => self::get_setting('message_position', 'right-top'),
                    'button_action_added' => self::get_setting('button_action_added', 'popup'),
                    'empty_confirm' => self::localization('empty_confirm', esc_html__('This action cannot be undone. Are you sure?', 'dejavo')),
                    'delete_confirm' => self::localization('delete_confirm', esc_html__('This action cannot be undone. Are you sure?', 'dejavo')),
                    'copied_text' => self::localization('copied', esc_html__('Copied the wishlist link:', 'dejavo')),
                    'menu_text' => '',
                    'button_text' => '<i class="dej-heart"></i>',
                    'button_text_added' => '<i class="dej-check"></i>',
                ]
            );
        }



        function get_items($key, $layout = null)
        {
            ob_start();
            // store $global_product
            global $product;
            $global_product = $product;
            $products = self::get_ids($key);
            $link = self::get_setting('link', 'yes');
            $table_tag = $tr_tag = $td_tag = 'div';
            $count = count($products); // count saved products
            $real_count = 0; // count real products
            $real_products = [];

            if ($layout === 'table') {
                $table_tag = 'table';
                $tr_tag = 'tr';
                $td_tag = 'td';
            }

            do_action('DejavoWL_before_items', $key, $products);

            if (is_array($products) && (count($products) > 0)) {
                echo '<' . $table_tag . ' class="DejavoWL-items">';
                do_action('DejavoWL_wishlist_items_before', $key, $products);

                foreach ($products as $product_id => $product_data) {
                    global $product;
                    $product = wc_get_product($product_id);

                    if (!$product || $product->get_status() !== 'publish') {
                        continue;
                    }

                    if (is_array($product_data) && isset($product_data['time'])) {
                        $product_time = date_i18n(get_option('date_format'), $product_data['time']);
                    } else {
                        // for old version
                        $product_time = date_i18n(get_option('date_format'), $product_data);
                    }


                    echo '<' . $tr_tag . ' class="' . esc_attr('DejavoWL-item DejavoWL-item-' . $product_id) . '" data-id="' . esc_attr($product_id) . '">';

                    if ($layout !== 'table') {
                        echo '<div class="DejavoWL-item-inner">';
                    }

                    do_action('DejavoWL_wishlist_item_before', $product, $key);

                    if (self::can_edit($key)) {
                        // remove
                        echo '<' . $td_tag . ' class="DejavoWL-item--remove"><span></span></' . $td_tag . '>';
                    }

                    // image
                    echo '<' . $td_tag . ' class="DejavoWL-item--image">';
                    do_action('DejavoWL_wishlist_item_image_before', $product, $key);

                    if ($link !== 'no') {
                        echo '<a ' . ($link === 'yes_popup' ? 'class="DejavoQV-link" data-id="' . esc_attr($product_id) . '" data-context="DejavoWL"' : '') . ' href="' . esc_url($product->get_permalink()) . '" ' . ($link === 'yes_blank' ? 'target="_blank"' : '') . '>';
                        echo wp_kses_post(apply_filters('DejavoWL_item_image', $product->get_image(), $product));
                        echo '</a>';
                    } else {
                        echo wp_kses_post(apply_filters('DejavoWL_item_image', $product->get_image(), $product));
                    }

                    do_action('DejavoWL_wishlist_item_image', $product, $key);
                    do_action('DejavoWL_wishlist_item_image_after', $product, $key);
                    echo '</' . $td_tag . '>';

                    // info
                    echo '<' . $td_tag . ' class="DejavoWL-item--info">';
                    do_action('DejavoWL_wishlist_item_info_before', $product, $key);

                    if ($link !== 'no') {
                        echo '<div class="DejavoWL-item--name"><a ' . ($link === 'yes_popup' ? 'class="DejavoQV-link" data-id="' . esc_attr($product_id) . '" data-context="DejavoWL"' : '') . ' href="' . esc_url($product->get_permalink()) . '" ' . ($link === 'yes_blank' ? 'target="_blank"' : '') . '>' . wp_kses_post(apply_filters('DejavoWL_item_name', $product->get_name(), $product)) . '</a></div>';
                    } else {
                        echo '<div class="DejavoWL-item--name">' . wp_kses_post(apply_filters('DejavoWL_item_name', $product->get_name(), $product)) . '</div>';
                    }

                    do_action('DejavoWL_wishlist_item_price_before', $product, $key);

                    echo '<div class="DejavoWL-item--price">' . wp_kses_post(apply_filters('DejavoWL_item_price', $product->get_price_html(), $product)) . '</div>';

                    if (self::get_setting('show_price_change', 'both') !== 'no') {
                        if (isset($product_data['price'])) {
                            $price = $product->get_price();

                            if ($price != $product_data['price']) {
                                // has price change
                                if ($price > $product_data['price']) {
                                    // increase
                                    $percentage = 100 * ($price - $product_data['price']) / $product_data['price'];
                                    $percentage = apply_filters('DejavoWL_price_increase_percentage', round($percentage) . '%', $percentage, $product_data);
                                    $increase = self::localization('price_increase', esc_html__('Increase {percentage} since added', 'dejavo'));
                                    $increase_mess = str_replace('{percentage}', $percentage, $increase);

                                    if (self::get_setting('show_price_change', 'both') === 'both' || self::get_setting('show_price_change', 'both') === 'increase') {
                                        echo '<div class="DejavoWL-item--price-change DejavoWL-item--price-increase">' . apply_filters('DejavoWL_price_increase_message', $increase_mess, $percentage, $product_data) . '</div>';
                                    }
                                }

                                if ($price < $product_data['price']) {
                                    // decrease
                                    $percentage = 100 * ($product_data['price'] - $price) / $product_data['price'];
                                    $percentage = apply_filters('DejavoWL_price_decrease_percentage', round($percentage) . '%', $percentage, $product_data);
                                    $decrease = self::localization('price_decrease', esc_html__('Decrease {percentage} since added', 'dejavo'));
                                    $decrease_mess = str_replace('{percentage}', $percentage, $decrease);

                                    if (self::get_setting('show_price_change', 'no') === 'both' || self::get_setting('show_price_change', 'both') === 'decrease') {
                                        echo '<div class="DejavoWL-item--price-change DejavoWL-item--price-decrease">' . apply_filters('DejavoWL_price_decrease_message', $decrease_mess, $percentage, $product_data) . '</div>';
                                    }
                                }
                            }
                        }
                    }

                    do_action('DejavoWL_wishlist_item_time_before', $product, $key);

                    echo '<div class="DejavoWL-item--time">' . esc_html(apply_filters('DejavoWL_item_time', $product_time, $product)) . '</div>';

                    do_action('DejavoWL_wishlist_item_info', $product, $key);
                    do_action('DejavoWL_wishlist_item_info_after', $product, $key);
                    echo '</' . $td_tag . '>';

                    // action
                    echo '<' . $td_tag . ' class="DejavoWL-item--actions">';
                    do_action('DejavoWL_wishlist_item_actions_before', $product, $key);

                    echo '<div class="DejavoWL-item--stock">' . apply_filters('DejavoWL_item_stock', wc_get_stock_html($product), $product) . '</div>';
                    echo '<div class="DejavoWL-item--add">' . apply_filters('DejavoWL_item_add_to_cart', do_shortcode('[add_to_cart style="" show_price="false" id="' . esc_attr($product_id) . '"]'), $product) . '</div>';

                    do_action('DejavoWL_wishlist_item_actions', $product, $key);
                    do_action('DejavoWL_wishlist_item_actions_after', $product, $key);
                    echo '</' . $td_tag . '>';

                    do_action('DejavoWL_wishlist_item_after', $product, $key);

                    if ($layout !== 'table') {
                        echo '</div><!-- /DejavoWL-item-inner -->';
                    }

                    echo '</' . $tr_tag . '>';

                    $real_products[$product_id] = $product_data;
                    $real_count++;

                }

                do_action('DejavoWL_wishlist_items_after', $key, $products);
                echo '</' . $table_tag . '>';
            } else {
                echo '<div class="DejavoWL-popup-content-mid-message">' . self::localization('empty_message', esc_html__('There are no products on the Wishlist!', 'dejavo')) . '</div>';
            }

            do_action('DejavoWL_after_items', $key, $products);


            // restore $global_product
            $product = $global_product;

            // update products
            if ($real_count < $count) {
                update_option('DejavoWL_list_' . $key, $real_products);
            }

            return apply_filters('DejavoWL_wishlist_items', ob_get_clean(), $key, $products);
        }


        function wp_footer()
        {
            if (is_admin()) {
                return;
            }

            echo '<div id="DejavoWL_wishlist" class="DejavoWL-popup ' . esc_attr('DejavoWL-popup-' . self::get_setting('popup_position', 'center')) . '"></div>';
        }

        function wishlist_content($key = false, $message = '')
        {
            if (empty($key)) {
                $key = self::get_key();
            }

            $products = self::get_ids($key);
            $count = count($products);
            $name = self::localization('popup_heading', esc_html__('Wishlist', 'dejavo'));

            ob_start();
            ?>
            <div class="DejavoWL-popup-inner" data-key="<?php echo esc_attr($key); ?>">
                <div class="DejavoWL-popup-content">
                    <div class="DejavoWL-popup-content-top">
                        <span class="DejavoWL-name"><?php echo esc_html($name); ?></span>
                        <?php
                        echo '<span class="DejavoWL-count-wrapper">';
                        echo '<span class="DejavoWL-count">' . esc_html($count) . '</span>';

                        if (self::get_setting('empty_button', 'yes') === 'yes') {
                            echo '<span class="DejavoWL-empty"' . ($count ? '' : ' style="display:none"') . '>' . self::localization('empty_button', esc_html__('remove all', 'dejavo')) . '</span>';
                        }

                        echo '</span>';
                        ?>
                        <span class="DejavoWL-popup-close"></span>
                    </div>
                    <div class="DejavoWL-popup-content-mid">
                        <?php if (!empty($message)) {
                            echo '<div class="DejavoWL-popup-content-mid-message">' . esc_html($message) . '</div>';
                        } else {
                            echo self::get_items($key);
                        } ?>
                    </div>
                    <div class="DejavoWL-popup-content-bot">
                        <div class="DejavoWL-popup-content-bot-inner">
                            <a class="DejavoWL-page" href="<?php echo esc_url(self::get_url($key, true)); ?>">
                                <?php echo self::localization('open_page', esc_html__('Open wishlist page', 'dejavo')); ?>
                            </a>
                            <a class="DejavoWL-continue"
                               href="<?php echo esc_url(self::get_setting('continue_url')); ?>"
                               data-url="<?php echo esc_url(self::get_setting('continue_url')); ?>">
                                <?php echo self::localization('continue', esc_html__('Continue shopping', 'dejavo')); ?>
                            </a>
                        </div>
                        <div class="DejavoWL-notice"></div>
                    </div>
                </div>
            </div>
            <?php
            return ob_get_clean();
        }

        function manage_content()
        {
            ?>
            <div class="DejavoWL-popup-inner">
                <div class="DejavoWL-popup-content">
                    <div class="DejavoWL-popup-content-top">
                        <?php echo self::localization('manage_wishlists', esc_html__('Manage wishlists', 'dejavo')); ?>
                        <span class="DejavoWL-popup-close"></span>
                    </div>
                    <div class="DejavoWL-popup-content-mid">
                        <?php if (($user_id = get_current_user_id())) { ?>
                            <table class="DejavoWL-items">
                                <?php
                                $key = get_user_meta($user_id, 'DejavoWL_key', true);
                                $keys = get_user_meta($user_id, 'DejavoWL_keys', true) ?: [];
                                $max = self::get_setting('maximum_wishlists', '5');

                                if (is_array($keys) && !empty($keys)) {
                                    foreach ($keys as $k => $wl) {
                                        $products = self::get_ids($k);
                                        $count = count($products);

                                        echo '<tr class="DejavoWL-item">';
                                        echo '<td>';

                                        if (isset($wl['type']) && ($wl['type'] === 'primary')) {
                                            echo '<a class="DejavoWL-view-wishlist" href="' . esc_url(self::get_url($k, true)) . '" data-key="' . esc_attr($k) . '">' . self::localization('primary_name', esc_html__('Wishlist', 'dejavo')) . '</a> - primary (' . $count . ')';
                                        } else {
                                            if (!empty($wl['name'])) {
                                                echo '<a class="DejavoWL-view-wishlist" href="' . esc_url(self::get_url($k, true)) . '" data-key="' . esc_attr($k) . '">' . $wl['name'] . '</a> (' . $count . ')';
                                            } else {
                                                echo '<a class="DejavoWL-view-wishlist" href="' . esc_url(self::get_url($k, true)) . '" data-key="' . esc_attr($k) . '">' . $k . '</a> (' . $count . ')';
                                            }
                                        }

                                        echo '</td><td style="text-align: end">';

                                        if ($key === $k) {
                                            echo '<span class="DejavoWL-default">' . self::localization('is_default', esc_html__('default', 'dejavo')) . '</span>';
                                        } else {
                                            echo '<a class="DejavoWL-set-default" data-key="' . esc_attr($k) . '" href="#">' . self::localization('set_default', esc_html__('set default', 'dejavo')) . '</a>';
                                        }

                                        echo '</td><td style="text-align: end">';

                                        if ((!isset($wl['type']) || ($wl['type'] !== 'primary')) && ($key !== $k)) {
                                            echo '<a class="DejavoWL-delete-wishlist" data-key="' . esc_attr($k) . '" href="#">' . self::localization('delete', esc_html__('delete', 'dejavo')) . '</a>';
                                        }

                                        echo '</td></tr>';
                                    }
                                }
                                ?>
                                <tr <?php echo(is_array($keys) && (count($keys) < (int)$max) ? '' : 'class="DejavoWL-disable"'); ?>>
                                    <td colspan="100%">
                                        <div class="DejavoWL-new-wishlist">
                                            <input type="text" id="DejavoWL_wishlist_name"
                                                   placeholder="<?php echo esc_attr(self::localization('placeholder_name', esc_html__('New Wishlist', 'dejavo'))); ?>"/>
                                            <input type="button" id="DejavoWL_add_wishlist"
                                                   value="<?php echo esc_attr(self::localization('add_wishlist', esc_html__('Add New Wishlist', 'dejavo'))); ?>"/>
                                        </div>
                                    </td>
                                </tr>
                            </table>
                        <?php } ?>
                    </div>
                </div>
            </div>
            <?php
        }

        function update_product_count($product_id, $action = 'add')
        {
            $meta_count = 'DejavoWL_count';
            $meta_time = ($action === 'add' ? 'DejavoWL_add' : 'DejavoWL_remove');
            $count = get_post_meta($product_id, $meta_count, true);
            $new_count = 0;

            if ($action === 'add') {
                if ($count) {
                    $new_count = absint($count) + 1;
                } else {
                    $new_count = 1;
                }
            } elseif ($action === 'remove') {
                if ($count && (absint($count) > 1)) {
                    $new_count = absint($count) - 1;
                } else {
                    $new_count = 0;
                }
            }

            update_post_meta($product_id, $meta_count, $new_count);
            update_post_meta($product_id, $meta_time, time());
        }

        public static function generate_key()
        {
            $key = '';
            $key_str = apply_filters('DejavoWL_key_characters', 'ABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789');
            $key_str_len = strlen($key_str);

            for ($i = 0; $i < apply_filters('DejavoWL_key_length', 6); $i++) {
                $key .= $key_str[random_int(0, $key_str_len - 1)];
            }

            return apply_filters('DejavoWL_generate_key', $key);
        }

        public static function can_edit($key)
        {
            if (is_user_logged_in()) {
                if (get_user_meta(get_current_user_id(), 'DejavoWL_key', true) === $key) {
                    return true;
                }

                if (($keys = get_user_meta(get_current_user_id(), 'DejavoWL_keys', true)) && isset($keys[$key])) {
                    return true;
                }
            } else {
                if (isset($_COOKIE['DejavoWL_key']) && (sanitize_text_field($_COOKIE['DejavoWL_key']) === $key)) {
                    return true;
                }
            }

            return false;
        }

        public static function get_page_id()
        {
            if (self::get_setting('page_id')) {
                return absint(self::get_setting('page_id'));
            }

            return false;
        }

        public static function get_key($new = false)
        {
            if ($new) {
                // get a new key for multiple wishlist
                $key = self::generate_key();

                while (self::exists_key($key)) {
                    $key = self::generate_key();
                }

                return $key;
            } else {
                if (!is_user_logged_in() && (self::get_setting('disable_unauthenticated', 'no') === 'yes')) {
                    return '#';
                }

                if (is_user_logged_in() && (($user_id = get_current_user_id()) > 0)) {
                    $key = get_user_meta($user_id, 'DejavoWL_key', true);

                    if (empty($key)) {
                        $key = self::generate_key();

                        while (self::exists_key($key)) {
                            $key = self::generate_key();
                        }

                        // set a new key
                        update_user_meta($user_id, 'DejavoWL_key', $key);

                        // multiple wishlist
                        update_user_meta($user_id, 'DejavoWL_keys', [
                            $key => [
                                'type' => 'primary',
                                'name' => '',
                                'time' => ''
                            ]
                        ]);
                    }

                    return $key;
                }

                if (isset($_COOKIE['DejavoWL_key'])) {
                    return trim(sanitize_text_field($_COOKIE['DejavoWL_key']));
                }

                return 'DejavoWL';
            }
        }

        public static function exists_key($key)
        {
            if (get_option('DejavoWL_list_' . $key)) {
                return true;
            }

            return false;
        }

        public static function get_ids($key = null)
        {
            if (!$key) {
                $key = self::get_key();
            }

            return (array)get_option('DejavoWL_list_' . $key, []);
        }

        public static function get_products()
        {
            return self::$products;
        }

        public static function get_url($key = null, $full = false)
        {
            $url = home_url('/');

            if ($page_id = self::get_page_id()) {
                if ($full) {
                    if (!$key) {
                        $key = self::get_key();
                    }

                    if (get_option('permalink_structure') !== '') {
                        $url = trailingslashit(get_permalink($page_id)) . $key;
                    } else {
                        $url = get_permalink($page_id) . '&DejavoWL_id=' . $key;
                    }
                } else {
                    $url = get_permalink($page_id);
                }
            }

            return esc_url(apply_filters('DejavoWL_wishlist_url', $url, $key));
        }

        public static function get_count($key = null)
        {
            if (!$key) {
                $key = self::get_key();
            }

            $products = self::get_ids($key);
            $count = count($products);

            return esc_html(apply_filters('DejavoWL_wishlist_count', $count, $key));
        }

        function product_columns($columns)
        {
            $columns['DejavoWL'] = esc_html__('Wishlist', 'dejavo');

            return $columns;
        }

        function posts_custom_column($column, $postid)
        {
            if ($column == 'DejavoWL') {
                if (($count = (int)get_post_meta($postid, 'DejavoWL_count', true)) > 0) {
                    echo '<a href="#" class="DejavoWL_action" data-pid="' . esc_attr($postid) . '">' . esc_html($count) . '</a>';
                }
            }
        }

        function sortable_columns($columns)
        {
            $columns['DejavoWL'] = 'DejavoWL';

            return $columns;
        }

        function request($vars)
        {
            if (isset($vars['orderby']) && 'DejavoWL' == $vars['orderby']) {
                $vars = array_merge($vars, [
                    'meta_key' => 'DejavoWL_count',
                    'orderby' => 'meta_value_num'
                ]);
            }

            return $vars;
        }

        function wp_login($user_login, $user)
        {
            if (isset($user->data->ID)) {
                $key = get_user_meta($user->data->ID, 'DejavoWL_key', true);

                if (empty($key)) {
                    $key = self::generate_key();

                    while (self::exists_key($key)) {
                        $key = self::generate_key();
                    }

                    // set a new key
                    update_user_meta($user->data->ID, 'DejavoWL_key', $key);
                }

                // multiple wishlist
                if (!get_user_meta($user->data->ID, 'DejavoWL_keys', true)) {
                    update_user_meta($user->data->ID, 'DejavoWL_keys', [
                        $key => [
                            'type' => 'primary',
                            'name' => '',
                            'time' => ''
                        ]
                    ]);
                }

                $secure = apply_filters('DejavoWL_cookie_secure', wc_site_is_https() && is_ssl());
                $httponly = apply_filters('DejavoWL_cookie_httponly', false);

                if (isset($_COOKIE['DejavoWL_key']) && !empty($_COOKIE['DejavoWL_key'])) {
                    wc_setcookie('DejavoWL_key_ori', trim(sanitize_text_field($_COOKIE['DejavoWL_key'])), time() + 604800, $secure, $httponly);
                }

                wc_setcookie('DejavoWL_key', $key, time() + 604800, $secure, $httponly);
            }
        }

        function wp_logout($user_id)
        {
            if (isset($_COOKIE['DejavoWL_key_ori']) && !empty($_COOKIE['DejavoWL_key_ori'])) {
                $secure = apply_filters('DejavoWL_cookie_secure', wc_site_is_https() && is_ssl());
                $httponly = apply_filters('DejavoWL_cookie_httponly', false);

                wc_setcookie('DejavoWL_key', trim(sanitize_text_field($_COOKIE['DejavoWL_key_ori'])), time() + 604800, $secure, $httponly);
            } else {
                unset($_COOKIE['DejavoWL_key_ori']);
                unset($_COOKIE['DejavoWL_key']);
            }
        }

        function display_post_states($states, $post)
        {
            if ('page' == get_post_type($post->ID) && $post->ID === absint(self::get_setting('page_id'))) {
                $states[] = esc_html__('Wishlist', 'dejavo');
            }

            return $states;
        }

        function users_columns($column)
        {
            $column['DejavoWL'] = esc_html__('Wishlist', 'dejavo');

            return $column;
        }

        function users_columns_content($val, $column_name, $user_id)
        {
            if ($column_name === 'DejavoWL') {
                $keys = get_user_meta($user_id, 'DejavoWL_keys', true);

                if (is_array($keys) && !empty($keys)) {
                    $val = '<a href="#" class="DejavoWL_action" data-uid="' . esc_attr($user_id) . '">' . count($keys) . '</a>';
                }
            }

            return $val;
        }

        function wishlist_quickview()
        {
            if (!check_ajax_referer('DejavoWL-nonce', 'nonce', false)) {
                die(esc_html__('Permissions check failed!', 'dejavo'));
            }

            global $wpdb;
            ob_start();
            echo '<div class="DejavoWL-quickview-items">';

            if (isset($_POST['key']) && $_POST['key'] != '') {
                $key = sanitize_text_field($_POST['key']);
                $products = self::get_ids($key);
                $count = count($products);

                if (count($products) > 0) {
                    $user = $wpdb->get_results($wpdb->prepare('SELECT user_id FROM `' . $wpdb->prefix . 'usermeta` WHERE `meta_key` = "DejavoWL_keys" AND `meta_value` LIKE "%s" LIMIT 1', '%"' . $key . '"%'));

                    echo '<div class="DejavoWL-quickview-item">';
                    echo '<div class="DejavoWL-quickview-item-image"><a href="' . esc_url(self::get_url($key, true)) . '" target="_blank">' . $key . '</a></div>';
                    echo '<div class="DejavoWL-quickview-item-info">';

                    if (!empty($user)) {
                        $user_id = $user[0]->user_id;
                        $user_data = get_userdata($user_id);

                        echo '<div class="DejavoWL-quickview-item-title"><a href="#" class="DejavoWL_action" data-uid="' . esc_attr($user_id) . '">' . $user_data->user_login . '</a></div>';
                        echo '<div class="DejavoWL-quickview-item-data">' . $user_data->user_email . ' | ' . sprintf(_n('%s product', '%s products', $count, 'dejavo'), number_format_i18n($count)) . '</div>';
                    } else {
                        echo '<div class="DejavoWL-quickview-item-title">' . esc_html__('Guest', 'dejavo') . '</div>';
                        echo '<div class="DejavoWL-quickview-item-data">' . sprintf(_n('%s product', '%s products', $count, 'dejavo'), number_format_i18n($count)) . '</div>';
                    }

                    echo '</div><!-- /DejavoWL-quickview-item-info -->';
                    echo '</div><!-- /DejavoWL-quickview-item -->';

                    foreach ($products as $pid => $data) {
                        if ($_product = wc_get_product($pid)) {
                            echo '<div class="DejavoWL-quickview-item">';
                            echo '<div class="DejavoWL-quickview-item-image">' . $_product->get_image() . '</div>';
                            echo '<div class="DejavoWL-quickview-item-info">';
                            echo '<div class="DejavoWL-quickview-item-title"><a href="' . get_edit_post_link($pid) . '" target="_blank">' . $_product->get_name() . '</a></div>';
                            echo '<div class="DejavoWL-quickview-item-data">' . date_i18n(get_option('date_format'), $data['time']) . ' <span class="DejavoWL-quickview-item-links">| ' . sprintf(esc_html__('Product ID: %s', 'dejavo'), $pid) . ' | <a href="#" class="DejavoWL_action" data-pid="' . esc_attr($pid) . '">' . esc_html__('See in wishlist', 'dejavo') . '</a></span></div>';
                            echo '</div><!-- /DejavoWL-quickview-item-info -->';
                            echo '</div><!-- /DejavoWL-quickview-item -->';
                        } else {
                            echo '<div class="DejavoWL-quickview-item">';
                            echo '<div class="DejavoWL-quickview-item-image">' . wc_placeholder_img() . '</div>';
                            echo '<div class="DejavoWL-quickview-item-info">';
                            echo '<div class="DejavoWL-quickview-item-title">' . sprintf(esc_html__('Product ID: %s', 'dejavo'), $pid) . '</div>';
                            echo '<div class="DejavoWL-quickview-item-data">' . esc_html__('This product is not available!', 'dejavo') . '</div>';
                            echo '</div><!-- /DejavoWL-quickview-item-info -->';
                            echo '</div><!-- /DejavoWL-quickview-item -->';
                        }
                    }
                } else {
                    echo '<div class="DejavoWL-quickview-item">';
                    echo '<div class="DejavoWL-quickview-item-image">' . wc_placeholder_img() . '</div>';
                    echo '<div class="DejavoWL-quickview-item-info">';
                    echo '<div class="DejavoWL-quickview-item-title">' . sprintf(esc_html__('Wishlist #%s', 'dejavo'), $key) . '</div>';
                    echo '<div class="DejavoWL-quickview-item-data">' . esc_html__('This wishlist have no product!', 'dejavo') . '</div>';
                    echo '</div><!-- /DejavoWL-quickview-item-info -->';
                    echo '</div><!-- /DejavoWL-quickview-item -->';
                }
            } elseif (isset($_POST['pid'])) {
                $pid = (int)sanitize_text_field($_POST['pid']);
                $per_page = (int)apply_filters('DejavoWL_quickview_per_page', 10);
                $page = isset($_POST['page']) ? abs((int)$_POST['page']) : 1;
                $offset = ($page - 1) * $per_page;
                $total = $wpdb->get_var($wpdb->prepare('SELECT COUNT(*) FROM `' . $wpdb->prefix . 'options` WHERE `option_name` LIKE "%DejavoWL_list_%" AND `option_value` LIKE "%s"', '%i:' . $pid . ';%'));
                $keys = $wpdb->get_results($wpdb->prepare('SELECT option_name FROM `' . $wpdb->prefix . 'options` WHERE `option_name` LIKE "%DejavoWL_list_%" AND `option_value` LIKE "%s" limit ' . $per_page . ' offset ' . $offset, '%i:' . $pid . ';%'));

                if ($total && is_countable($keys) && count($keys)) {
                    echo '<div class="DejavoWL-quickview-item">';

                    if ($_product = wc_get_product($pid)) {
                        echo '<div class="DejavoWL-quickview-item-image">' . $_product->get_image() . '</div>';
                        echo '<div class="DejavoWL-quickview-item-info">';
                        echo '<div class="DejavoWL-quickview-item-title"><a href="' . get_edit_post_link($pid) . '" target="_blank">' . $_product->get_name() . '</a></div>';
                        echo '<div class="DejavoWL-quickview-item-data">' . sprintf(esc_html__('Product ID: %s', 'dejavo'), $pid) . ' | ' . sprintf(_n('%s wishlist', '%s wishlists', $total, 'DejavoWL'), number_format_i18n($total)) . '</div>';
                    } else {
                        echo '<div class="DejavoWL-quickview-item-image">' . wc_placeholder_img() . '</div>';
                        echo '<div class="DejavoWL-quickview-item-info">';
                        echo '<div class="DejavoWL-quickview-item-title">' . sprintf(esc_html__('Product ID: %s', 'dejavo'), $pid) . '</div>';
                        echo '<div class="DejavoWL-quickview-item-data">' . esc_html__('This product is not available!', 'dejavo') . '</div>';
                    }

                    // paging
                    if ($total > $per_page) {
                        $pages = ceil($total / $per_page);
                        echo '<div class="DejavoWL-quickview-item-paging">Page ';

                        echo '<select class="DejavoWL_paging" data-pid="' . $pid . '">';

                        for ($i = 1; $i <= $pages; $i++) {
                            echo '<option value="' . $i . '" ' . selected($page, $i, false) . '>' . $i . '</option>';
                        }

                        echo '</select> / ' . $pages;

                        echo '</div><!-- /DejavoWL-quickview-item-paging -->';
                    }

                    echo '</div><!-- /DejavoWL-quickview-item-info -->';
                    echo '</div><!-- /DejavoWL-quickview-item -->';

                    foreach ($keys as $item) {
                        $products = get_option($item->option_name);
                        $products_count = count($products);
                        $key = str_replace('DejavoWL_list_', '', $item->option_name);
                        $user = $wpdb->get_results($wpdb->prepare('SELECT user_id FROM `' . $wpdb->prefix . 'usermeta` WHERE `meta_key` = "DejavoWL_keys" AND `meta_value` LIKE "%s" LIMIT 1', '%"' . $key . '"%'));

                        echo '<div class="DejavoWL-quickview-item">';
                        echo '<div class="DejavoWL-quickview-item-image"><a href="' . esc_url(self::get_url($key, true)) . '" target="_blank">' . esc_html($key) . '</a></div>';
                        echo '<div class="DejavoWL-quickview-item-info">';

                        if (!empty($user)) {
                            $user_id = $user[0]->user_id;
                            $user_data = get_userdata($user_id);

                            echo '<div class="DejavoWL-quickview-item-title"><a href="#" class="DejavoWL_action" data-uid="' . esc_attr($user_id) . '">' . $user_data->user_login . '</a></div>';
                            echo '<div class="DejavoWL-quickview-item-data">' . $user_data->user_email . '  | <a href="#" class="DejavoWL_action DejavoWL_action_' . $products_count . '" data-key="' . esc_attr($key) . '">' . sprintf(_n('%s product', '%s products', $products_count, 'dejavo'), number_format_i18n($products_count)) . '</a></div>';
                        } else {
                            echo '<div class="DejavoWL-quickview-item-title">' . esc_html__('Guest', 'dejavo') . '</div>';
                            echo '<div class="DejavoWL-quickview-item-data"><a href="#" class="DejavoWL_action" data-key="' . esc_attr($key) . '">' . sprintf(_n('%s product', '%s products', $products_count, 'dejavo'), number_format_i18n($products_count)) . '</a></div>';
                        }

                        echo '</div><!-- /DejavoWL-quickview-item-info -->';
                        echo '</div><!-- /DejavoWL-quickview-item -->';
                    }
                }
            } elseif (isset($_POST['uid'])) {
                $user_id = (int)sanitize_text_field($_POST['uid']);
                $keys = get_user_meta($user_id, 'DejavoWL_keys', true) ?: [];

                if ($user = get_user_by('id', $user_id)) {
                    echo '<div class="DejavoWL-quickview-item">';
                    echo '<div class="DejavoWL-quickview-item-image"><img src="' . esc_url(get_avatar_url($user_id)) . '" /></div>';
                    echo '<div class="DejavoWL-quickview-item-info">';
                    echo '<div class="DejavoWL-quickview-item-title"><a href="' . get_edit_user_link($user_id) . '" target="_blank">' . $user->user_login . '</a></div>';
                    echo '<div class="DejavoWL-quickview-item-data">' . $user->user_email . '</div>';
                    echo '</div><!-- /DejavoWL-quickview-item-info -->';
                    echo '</div><!-- /DejavoWL-quickview-item -->';
                }

                if (is_array($keys) && count($keys)) {
                    foreach ($keys as $key => $data) {
                        $products = self::get_ids($key);
                        $products_count = count($products);

                        echo '<div class="DejavoWL-quickview-item">';
                        echo '<div class="DejavoWL-quickview-item-image"><a href="' . esc_url(self::get_url($key, true)) . '" target="_blank">' . $key . '</a></div>';
                        echo '<div class="DejavoWL-quickview-item-info">';
                        echo '<div class="DejavoWL-quickview-item-title">' . (!empty($data['name']) ? $data['name'] : 'Primary') . '</div>';
                        echo '<div class="DejavoWL-quickview-item-data"><a href="#" class="DejavoWL_action DejavoWL_action_' . $products_count . '" data-key="' . esc_attr($key) . '">' . sprintf(_n('%s product', '%s products', $products_count, 'dejavo'), number_format_i18n($products_count)) . '</a></div>';
                        echo '</div><!-- /DejavoWL-quickview-item-info -->';
                        echo '</div><!-- /DejavoWL-quickview-item -->';
                    }
                }
            }

            echo '</div><!-- /DejavoWL-quickview-items -->';
            echo apply_filters('DejavoWL_wishlist_quickview', ob_get_clean());
            die();
        }


        function wcml_multi_currency($ajax_actions)
        {
            $ajax_actions[] = 'view_wishlist';
            $ajax_actions[] = 'wishlist_add';
            $ajax_actions[] = 'wishlist_remove';
            $ajax_actions[] = 'wishlist_load';
            $ajax_actions[] = 'DejavoWL_get_data';

            return $ajax_actions;
        }


        function get_data()
        {
            $data = [
                'key' => self::get_key(),
                'ids' => self::get_ids(),
            ];

            wp_send_json($data);
        }
    }

    return DejavoWL::instance();


}

